/*
 * prodver.c -  return Product ID and Revision from Crestron platform
 *
 *  NOTE:  This thread is not used to set global product ID, since it is
 * 		not guaranteed to be executed in time. thread is only to
 *		handle uiHAL interface. Global product ID is set in platform file.
 *
 * Copyright (C) Crestron Electronics
 *
 * Written: RJK
 */

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#include <mach/gpio.h>

#define PROC_PRODUCT_DIR	"product"
#define PROC_PROD_ID_FILE 	"prodID"
#define PROCFS_MAX_SIZE     2048

static u32 gProdID=0;  // this parameter gets set once at init, then referenced on reads
static u32 gRevID=0;  // this parameter gets set once at init, then referenced on reads

/* tst600 product ID pins */
#define PROD_ID5                        (5*32 + 12)     /* GPIO_6_12 */
#define PROD_ID4                        (5*32 + 11)     /* GPIO_6_11 */
#define PROD_ID3                        (5*32 + 10)     /* GPIO_6_10 */
#define PROD_ID2                        (5*32 + 9)      /* GPIO_6_9 */
#define PROD_ID1                        (5*32 + 8)      /* GPIO_6_8 */
#define PROD_ID0                        (5*32 + 7)      /* GPIO_6_7 */

/* tst600 product ID pins */
#define PROD_REV3                       (5*32 + 16)     /* GPIO_6_16 */
#define PROD_REV2                       (5*32 + 15)     /* GPIO_6_15 */
#define PROD_REV1                       (5*32 + 14)     /* GPIO_6_14 */
#define PROD_REV0                       (5*32 + 13)     /* GPIO_6_13 */

/**
 * The structure keeping information about the /proc file
 *
 */
static struct proc_dir_entry *prod_dir, *prod_file;

/**
 * The buffer (2k) for this module
 *
 */
static char procfs_buffer_prod[PROCFS_MAX_SIZE];

/**
 * The size of the data held in the buffer
 *
 */
static unsigned long procfs_buffer_size_prod = 0;

/**
 * This funtion is called when the /proc file is read
 *
 */
static ssize_t procfs_read_prodID(struct file *filep,	/* see include/linux/fs.h   */
			     char *buffer,	/* buffer to fill with data */
			     size_t length,	/* length of the buffer     */
			     loff_t * offset)
{
   char * p = procfs_buffer_prod;
   static int finished = 0;

   /* needed to stop from continuously printing */ 
   if ( finished == 1 ) { finished=0; return 0; }
   finished = 1;

   p += sprintf ( p, "PRODUCT_ID=%04X\n" , gProdID );
   p += sprintf ( p, "REVISION_ID=%04X\n", gRevID );

   procfs_buffer_size_prod = p-procfs_buffer_prod;

	if ( copy_to_user(buffer, procfs_buffer_prod, procfs_buffer_size_prod) ) {
		return -EFAULT;
	}

  return procfs_buffer_size_prod;
}
static ssize_t procfs_write_prodID(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	printk(KERN_ERR "Can't write to prodid\n");
	return -EFAULT;
}

/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission_prod(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int procfs_open_prod(struct inode *inode, struct file *file)
{
	try_module_get(THIS_MODULE);
	return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int procfs_close_prod(struct inode *inode, struct file *file)
{
	module_put(THIS_MODULE);
	return 0;		/* success */
}

static struct file_operations File_Ops_Prod_File = {
	.read 	 = procfs_read_prodID,
	.write 	 = procfs_write_prodID,
	.open 	 = procfs_open_prod,
	.release = procfs_close_prod,
};

/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_Prod_File = {
	.permission = module_permission_prod,	/* check for permissions */
};

/* productVersion_init function: this function returns product ID and Revision ID */
static int productVersion_init()
{
        int id=0;
        int bits=0; 
        int x=0; 
	
	/* create directory */
	prod_dir = proc_mkdir(PROC_PRODUCT_DIR, NULL);
	if(prod_dir == NULL) {
		return -ENOMEM;
	}
	//prod_dir->owner = THIS_MODULE;

 	/* create the /proc file */
	prod_file = create_proc_entry(PROC_PROD_ID_FILE, 0644, prod_dir);
	/* check if the /proc file was created successfuly */
	if (prod_file == NULL){
		printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
		       PROC_PROD_ID_FILE);
		return -ENOMEM;
	}
	else
	{
	  //prod_file->owner = THIS_MODULE;
	  prod_file->proc_iops = &Inode_Ops_Prod_File;
	  prod_file->proc_fops = &File_Ops_Prod_File;
	  prod_file->mode = S_IFREG | S_IRUGO | S_IWUSR;
	  prod_file->uid = 0;
	  prod_file->gid = 0;
	  prod_file->size = 80;
       }

	// Read product ID and revision ID from mx53
        for(x=0;x<6;x++) {
                bits=gpio_get_value(PROD_ID0+x);
                id |= (bits << x);
        }
        gProdID=id;

	id=0;
        for(x=0;x<4;x++) {
                bits=gpio_get_value(PROD_REV0+x);
                id |= (bits << x);
        }
        gRevID=id;

	printk("Product ID set for uiHAL interface: ID\/REV : 0x%x\/0x%x\n", gProdID, gRevID);

	return 0;

}

static int __init prodver_init(void)
{
	int ret;

	ret = productVersion_init();	
	return ret;
}
module_init(prodver_init);

static void __exit prodver_exit(void)
{
	//exit
}
module_exit(prodver_exit);

MODULE_AUTHOR("Raymond Kuschan");
MODULE_DESCRIPTION("Crestron Driver");
MODULE_LICENSE("GPL v2");
